/**
	=========================================================
	GREEDY RECTANGLE MERGING & PERIMETER EXTRACTION
	=========================================================

	Pipeline:
	1) binary grid (0/1)
	2) greedy rectangle merging
	3) external perimeter extraction (from grid)
	4) collinear segment merging
	5) closed loop construction
	6) orthogonal loop expansion (optional)

	This module is framework-agnostic and purely geometric.
	=========================================================
*/

/**
 * A 2D point expressed in grid or world units.
 */
export interface Point {
	x: number;
	y: number;
}

/**
 * Axis-aligned rectangle in grid units.
 */
export interface Rect {
	x: number;
	y: number;
	w: number;
	h: number;
}

/**
 * Axis-aligned line segment.
 */
export interface Segment {
	x1: number;
	y1: number;
	x2: number;
	y2: number;
}

/**
 * Result of the greedy perimeter extraction pipeline.
 */
export interface GreedyResult {
	rectangles: Rect[];
	loops: Point[][];
	mode: 'horizontal-first' | 'vertical-first';
}

/**
 * Result of a greedy rectangle merge pass.
 */
export interface GreedyMergeResult {
	rectangles: Rect[];
	mode: 'horizontal-first' | 'vertical-first';
}

/**
 * Greedy geometry utilities.
 *
 * This class provides a full pipeline to:
 * - reduce a binary grid into rectangles
 * - extract the external perimeter
 * - merge collinear edges
 * - reconstruct closed polygonal loops
 *
 * All operations are deterministic, grid-based,
 * and independent from rendering or physics engines.
 */
export class Greedy {

	/**
	 * Performs greedy rectangle merging on a binary grid.
	 *
	 * The grid is scanned top-left to bottom-right.
	 * When a filled cell is found, the algorithm expands
	 * a maximal axis-aligned rectangle according to the
	 * chosen strategy, marking all consumed cells as visited.
	 *
	 * @param grid 2D array of 0/1 values
	 * @param mode Expansion strategy
	 * @returns Array of non-overlapping rectangles
	 */
	static merge(grid: number[][], mode: 'horizontal-first' | 'vertical-first'): Rect[] {

		const height: number = grid.length;
		const width: number = grid[0].length;

		const visited: boolean[][] = [];

		for (let i: number = 0; i < height; i++) {
			visited[i] = new Array<boolean>(width).fill(false);
		}

		const rects: Rect[] = [];

		for (let i: number = 0; i < height; i++) {
			for (let j: number = 0; j < width; j++) {

				if (visited[i][j] || grid[i][j] === 0) {
					visited[i][j] = true;
					continue;
				}

				let w: number = 1;
				let h: number = 1;

				if (mode === 'horizontal-first') {

					while (j + w < width && !visited[i][j + w] && grid[i][j + w] === 1) {
						w++;
					}

					while (i + h < height) {

						let ok: boolean = true;

						for (let jj: number = j; jj < j + w; jj++) {
							if (visited[i + h][jj] || grid[i + h][jj] === 0) {
								ok = false;
								break;
							}
						}

						if (!ok) {
							break;
						}

						h++;
					}
				}
				else {

					while (i + h < height && !visited[i + h][j] && grid[i + h][j] === 1) {
						h++;
					}

					while (j + w < width) {

						let ok: boolean = true;

						for (let ii: number = i; ii < i + h; ii++) {
							if (visited[ii][j + w] || grid[ii][j + w] === 0) {
								ok = false;
								break;
							}
						}

						if (!ok) {
							break;
						}

						w++;
					}
				}

				for (let ii: number = i; ii < i + h; ii++) {
					for (let jj: number = j; jj < j + w; jj++) {
						visited[ii][jj] = true;
					}
				}

				rects.push({
					x: j,
					y: i,
					w: w,
					h: h
				});
			}
		}

		return rects;
	}

	/**
	 * Runs both merge strategies and keeps the one
	 * that produces fewer rectangles.
	 *
	 * @param grid Binary grid
	 * @returns Best merge result and selected strategy
	 */
	static mergeBest(grid: number[][]): GreedyMergeResult {

		const horizontal: Rect[] = Greedy.merge(grid, 'horizontal-first');
		const vertical: Rect[] = Greedy.merge(grid, 'vertical-first');

		if (vertical.length < horizontal.length) {
			return {
				rectangles: vertical,
				mode: 'vertical-first'
			};
		}

		return {
			rectangles: horizontal,
			mode: 'horizontal-first'
		};
	}

	/**
	 * Extracts the true external perimeter directly from a binary grid.
	 *
	 * Internal edges shared by adjacent filled cells are discarded,
	 * leaving only the external boundary.
	 *
	 * @param grid 2D array of 0/1 values
	 * @returns Array of perimeter segments
	 */
	static buildPerimeterFromGrid(grid: number[][]): Segment[] {

		const height: number = grid.length;
		const width: number = grid[0].length;

		const segments: Segment[] = [];

		function filled(i: number, j: number): boolean {
			if (i < 0 || j < 0 || i >= height || j >= width) {
				return false;
			}
			return grid[i][j] === 1;
		}

		for (let i: number = 0; i < height; i++) {
			for (let j: number = 0; j < width; j++) {

				if (grid[i][j] === 0) {
					continue;
				}

				if (!filled(i - 1, j)) {
					segments.push({ x1: j, y1: i, x2: j + 1, y2: i });
				}

				if (!filled(i + 1, j)) {
					segments.push({ x1: j, y1: i + 1, x2: j + 1, y2: i + 1 });
				}

				if (!filled(i, j - 1)) {
					segments.push({ x1: j, y1: i, x2: j, y2: i + 1 });
				}

				if (!filled(i, j + 1)) {
					segments.push({ x1: j + 1, y1: i, x2: j + 1, y2: i + 1 });
				}
			}
		}

		return segments;
	}

	/**
	 * Merges collinear and adjacent perimeter segments.
	 *
	 * @param segments Raw perimeter segments
	 * @returns Simplified segment list
	 */
	static mergeCollinearSegments(segments: Segment[]): Segment[] {

		const horizontal: Map<number, Segment[]> = new Map<number, Segment[]>();
		const vertical: Map<number, Segment[]> = new Map<number, Segment[]>();

		for (const s of segments) {

			if (s.y1 === s.y2) {

				if (!horizontal.has(s.y1)) {
					horizontal.set(s.y1, []);
				}

				horizontal.get(s.y1)!.push(s);
			}
			else {

				if (!vertical.has(s.x1)) {
					vertical.set(s.x1, []);
				}

				vertical.get(s.x1)!.push(s);
			}
		}

		const out: Segment[] = [];

		for (const list of horizontal.values()) {

			list.sort((a: Segment, b: Segment) => a.x1 - b.x1);

			let current: Segment = { ...list[0] };

			for (let i: number = 1; i < list.length; i++) {

				if (current.x2 === list[i].x1) {
					current.x2 = list[i].x2;
				}
				else {
					out.push(current);
					current = { ...list[i] };
				}
			}

			out.push(current);
		}

		for (const list of vertical.values()) {

			list.sort((a: Segment, b: Segment) => a.y1 - b.y1);

			let current: Segment = { ...list[0] };

			for (let i: number = 1; i < list.length; i++) {

				if (current.y2 === list[i].y1) {
					current.y2 = list[i].y2;
				}
				else {
					out.push(current);
					current = { ...list[i] };
				}
			}

			out.push(current);
		}

		return out;
	}

	/**
	 * Reconstructs closed polygonal loops from perimeter segments.
	 *
	 * @param segments Merged perimeter segments
	 * @returns Closed loops as ordered vertex arrays
	 */
	static buildLoops(segments: Segment[]): Point[][] {

		const adjacency: Map<string, Point[]> = new Map<string, Point[]>();

		function key(p: Point): string {
			return `${p.x},${p.y}`;
		}

		function addEdge(a: Point, b: Point): void {

			const ka: string = key(a);
			const kb: string = key(b);

			if (!adjacency.has(ka)) {
				adjacency.set(ka, []);
			}

			if (!adjacency.has(kb)) {
				adjacency.set(kb, []);
			}

			adjacency.get(ka)!.push(b);
			adjacency.get(kb)!.push(a);
		}

		for (const s of segments) {
			addEdge({ x: s.x1, y: s.y1 }, { x: s.x2, y: s.y2 });
		}

		const loops: Point[][] = [];
		const usedEdges: Set<string> = new Set<string>();

		function edgeKey(a: Point, b: Point): string {
			return `${key(a)}-${key(b)}`;
		}

		for (const [k, neighbors] of adjacency.entries()) {

			for (const n of neighbors) {

				const start: Point = Greedy.parsePoint(k);
				const ek: string = edgeKey(start, n);

				if (usedEdges.has(ek)) {
					continue;
				}

				const loop: Point[] = [start];
				let current: Point = start;
				let previous: Point | null = null;

				while (true) {

					const candidates: Point[] = adjacency.get(key(current))!;
					let next: Point | null = null;

					for (const c of candidates) {

						if (previous && c.x === previous.x && c.y === previous.y) {
							continue;
						}

						next = c;
						break;
					}

					if (!next) {
						break;
					}

					usedEdges.add(edgeKey(current, next));
					usedEdges.add(edgeKey(next, current));

					previous = current;
					current = next;

					if (current.x === start.x && current.y === start.y) {
						break;
					}

					loop.push(current);
				}

				if (loop.length > 2) {
					loops.push(loop);
				}
			}
		}

		return loops;
	}

	/**
	 * Expands an orthogonal closed loop outward by a fixed offset.
	 *
	 * @param loop Closed orthogonal loop
	 * @param offset Expansion distance
	 * @returns Expanded loop
	 */
	static expandOrthogonalLoop(loop: Point[], offset: number): Point[] {

		const count: number = loop.length;
		const out: Point[] = [];

		function direction(a: Point, b: Point): 'right' | 'left' | 'up' | 'down' {

			if (b.x > a.x) {
				return 'right';
			}

			if (b.x < a.x) {
				return 'left';
			}

			if (b.y > a.y) {
				return 'down';
			}

			return 'up';
		}

		function normal(d: 'right' | 'left' | 'up' | 'down'): Point {

			switch (d) {
				case 'right': return { x: 0, y: -offset };
				case 'left':  return { x: 0, y:  offset };
				case 'down':  return { x:  offset, y: 0 };
				case 'up':    return { x: -offset, y: 0 };
			}
		}

		for (let i: number = 0; i < count; i++) {

			const prev: Point = loop[(i - 1 + count) % count];
			const curr: Point = loop[i];
			const next: Point = loop[(i + 1) % count];

			const d1 = direction(prev, curr);
			const d2 = direction(curr, next);

			const n1: Point = normal(d1);
			const n2: Point = normal(d2);

			out.push({
				x: curr.x + n1.x + n2.x,
				y: curr.y + n1.y + n2.y
			});
		}

		return out;
	}

    /**
     * Full greedy perimeter extraction pipeline.
     *
     * @param grid Binary grid
     * @returns Rectangles, loops and selected strategy
     */
    static extractLoops(grid: number[][]): GreedyResult {

        const merge: GreedyMergeResult = Greedy.mergeBest(grid);
        const raw: Segment[] = Greedy.buildPerimeterFromGrid(grid);
        const merged: Segment[] = Greedy.mergeCollinearSegments(raw);
        const loops: Point[][] = Greedy.buildLoops(merged);

        const result: GreedyResult = {
            rectangles: merge.rectangles,
            loops: loops,
            mode: merge.mode
        };

        return result;
    }

	/**
	 * Parses a point from a string key in the form "x,y".
	 */
	private static parsePoint(key: string): Point {

		const parts: number[] = key.split(',').map(Number);

		return {
			x: parts[0],
			y: parts[1]
		};
	}
}