/**
 * =========================================================
 * PATH ANALYZER
 * ---------------------------------------------------------
 * Takes a small (true perimeter) loop and its expanded
 * counterpart, and produces motion-ready path segments
 * with precomputed length and terrain direction.
 *
 * This is pure geometry + semantics.
 * =========================================================
 */

import { Point } from './greedy';

/**
 * Cardinal directions, orthogonal only.
 */
export enum OrthogonalDirection {
	UP,
	RIGHT,
	DOWN,
	LEFT
}

/**
 * Segment used for movement.
 * All values are in PIXEL space.
 */
export interface MotionSegment {
	start: Point;
	end: Point;
	length: number;
	terrain: OrthogonalDirection;
}

/**
 * Utility class to convert perimeter loops into
 * motion-ready segments.
 */
export class PathAnalyzer {

	/**
	 * Analyzes a perimeter loop and its expanded version,
	 * producing motion segments with terrain direction.
	 *
	 * @param smallLoop    Original perimeter (tile or pixel space)
	 * @param expandedLoop Expanded perimeter (same space as smallLoop)
	 */
	static analyze(
		smallLoop: Point[],
		expandedLoop: Point[]
	): MotionSegment[] {

		const count: number = smallLoop.length;
		const segments: MotionSegment[] = [];

		for (let i: number = 0; i < count; i++) {

			const aSmall: Point = smallLoop[i];
			const bSmall: Point = smallLoop[(i + 1) % count];

			const aBig: Point = expandedLoop[i];
			const bBig: Point = expandedLoop[(i + 1) % count];

			const orientation: 'horizontal' | 'vertical' =
				PathAnalyzer.getOrientation(aSmall, bSmall);

			const terrain: OrthogonalDirection =
				PathAnalyzer.getTerrainDirection(
					aSmall,
					bSmall,
					aBig,
					bBig,
					orientation
				);

			const dx: number = bBig.x - aBig.x;
			const dy: number = bBig.y - aBig.y;
			const length: number = Math.hypot(dx, dy);

			segments.push({
				start: { x: aBig.x, y: aBig.y },
				end: { x: bBig.x, y: bBig.y },
				length: length,
				terrain: terrain
			});
		}

		return segments;
	}

	/**
	 * Determines if a segment is horizontal or vertical.
	 */
	private static getOrientation(
		a: Point,
		b: Point
	): 'horizontal' | 'vertical' {

		if (a.y === b.y) {
			return 'horizontal';
		}

		return 'vertical';
	}

	/**
	 * Determines on which side the terrain lies by comparing
	 * the midpoint of the small and expanded segments.
	 */
	private static getTerrainDirection(
		aSmall: Point,
		bSmall: Point,
		aBig: Point,
		bBig: Point,
		orientation: 'horizontal' | 'vertical'
	): OrthogonalDirection {

		const midSmallX: number = (aSmall.x + bSmall.x) * 0.5;
		const midSmallY: number = (aSmall.y + bSmall.y) * 0.5;

		const midBigX: number = (aBig.x + bBig.x) * 0.5;
		const midBigY: number = (aBig.y + bBig.y) * 0.5;

		if (orientation === 'horizontal') {

			if (midBigY < midSmallY) {
				return OrthogonalDirection.UP;
			}

			return OrthogonalDirection.DOWN;
		}

		// vertical
		if (midBigX < midSmallX) {
			return OrthogonalDirection.LEFT;
		}

		return OrthogonalDirection.RIGHT;
	}
}
