// THE GAME ITSELF

import { GameOptions } from '../gameOptions';
import { Greedy, Point } from '../geometry/greedy';
import { PathAnalyzer, MotionSegment, OrthogonalDirection } from '../geometry/pathAnalyzer';

enum walkingDirection {
	CLOCKWISE,
	COUNTERCLOCKWISE
}

interface LineIntersection {
    point: Phaser.Math.Vector2;
    t1: number;   
    t2: number;   
}

export class PlayGame extends Phaser.Scene {

	constructor() {
		super('PlayGame');
	}

	hero: Phaser.GameObjects.Sprite;
	layer: Phaser.Tilemaps.TilemapLayer | Phaser.Tilemaps.TilemapGPULayer;
    controls: Phaser.Types.Input.Keyboard.CursorKeys;
    pathIndex: number;
	segmentIndex: number;
	segmentT: number;
	direction: walkingDirection ;
    allPathSegments: MotionSegment[][];
    isJumping: boolean;
    collisionPerimeterLines: Phaser.Geom.Line[][] = [];
    gravityVector: Phaser.Math.Vector2;
    jumpVector: Phaser.Math.Vector2;
    skipCollision: boolean;

	create(): void {

        this.controls = this.input.keyboard!.createCursorKeys();

		const map: Phaser.Tilemaps.Tilemap = this.make.tilemap({ key: 'map' });
		const tileset: Phaser.Tilemaps.Tileset = map.addTilesetImage('tile', 'tile')!;
		this.layer = map.createLayer(0, tileset, 0, 0);

		const grid: number[][] = [];

		for (let i: number = 0; i < map.height; i++) {
			grid[i] = [];
			for (let j: number = 0; j < map.width; j++) {
				const tile: Phaser.Tilemaps.Tile = this.layer.getTileAt(j, i);
				grid[i][j] = tile && tile.index === 1 ? 1 : 0;
			}
		}

        const greedyResult = Greedy.extractLoops(grid);
        this.allPathSegments = [];
        
        for (let i: number = 0; i < greedyResult.loops.length; i++) {

            this.collisionPerimeterLines[i] = [];

            const expandedLoop: Point[] = Greedy.expandOrthogonalLoop(greedyResult.loops[i], 0.25);
            const pathSegments: MotionSegment[] = PathAnalyzer.analyze(greedyResult.loops[i], expandedLoop);
		    this.allPathSegments.push(pathSegments);

            for (let j: number = 0; j < expandedLoop.length; j++) {
                const aBig: Point = expandedLoop[j];
		        const bBig: Point = expandedLoop[(j + 1) % expandedLoop.length];
                const line: Phaser.Geom.Line = new Phaser.Geom.Line(aBig.x * GameOptions.tileSize, aBig.y * GameOptions.tileSize, bBig.x * GameOptions.tileSize, bBig.y * GameOptions.tileSize)
                this.collisionPerimeterLines[i].push(line)
            }   
        }

        this.pathIndex = 1
        this.segmentIndex = 0;
        this.segmentT = 0;
        this.direction = walkingDirection.CLOCKWISE;

        this.hero = this.add.sprite(0,0, 'hero');
        this.placeOnSegment();

        this.input.on('pointerdown', () => {
            this.direction = this.direction === walkingDirection.CLOCKWISE ? walkingDirection.COUNTERCLOCKWISE : walkingDirection.CLOCKWISE;
        });

        this.input.keyboard!.on('keydown-SPACE', () => {   
		    this.jump();
        });
	}

    private placeOnSegment(): void {
	    const path: MotionSegment[] = this.allPathSegments[this.pathIndex];
        const segment: MotionSegment = path[this.segmentIndex];
	    const posX: number = Phaser.Math.Linear(segment.start.x, segment.end.x, this.segmentT) * GameOptions.tileSize;
        const posY: number = Phaser.Math.Linear(segment.start.y, segment.end.y, this.segmentT) * GameOptions.tileSize;
        this.hero.setPosition(posX, posY);
        switch (segment.terrain) {
            case OrthogonalDirection.UP:
			    this.hero.setRotation(0);
			    this.hero.setFlipX(this.direction === walkingDirection.COUNTERCLOCKWISE);
			    break;
            case OrthogonalDirection.RIGHT:
			    this.hero.setRotation(Math.PI / 2);
                this.hero.setFlipX(this.direction === walkingDirection.COUNTERCLOCKWISE);
		    	break;
            case OrthogonalDirection.DOWN:
			    this.hero.setRotation(Math.PI);
                this.hero.setFlipX(this.direction === walkingDirection.COUNTERCLOCKWISE);
			    break;
            case OrthogonalDirection.LEFT:
			    this.hero.setRotation(-Math.PI / 2);
			    this.hero.setFlipX(this.direction === walkingDirection.COUNTERCLOCKWISE);
			    break;
	    }
    }

    private jump(): void {
        if (this.isJumping) {
            return;
        }
        this.isJumping = true;
        this.skipCollision = true;
        const segment: MotionSegment = this.allPathSegments[this.pathIndex][this.segmentIndex];
        const walkingSpeed: number = GameOptions.heroSpeed * (this.direction === walkingDirection.CLOCKWISE ? 1 : -1);
        switch (segment.terrain) {
            case OrthogonalDirection.UP:
                this.gravityVector = new Phaser.Math.Vector2(0, GameOptions.gameGravity);
                this.jumpVector = new Phaser.Math.Vector2(walkingSpeed, -GameOptions.jumpForce);
                break;
            case OrthogonalDirection.RIGHT:
                this.gravityVector = new Phaser.Math.Vector2(-GameOptions.gameGravity, 0);
                this.jumpVector = new Phaser.Math.Vector2(GameOptions.jumpForce, walkingSpeed);
                break;
            case OrthogonalDirection.DOWN:
                this.gravityVector = new Phaser.Math.Vector2(0, -GameOptions.gameGravity);
                this.jumpVector = new Phaser.Math.Vector2(-walkingSpeed, GameOptions.jumpForce);
                break;
            case OrthogonalDirection.LEFT:
                this.gravityVector = new Phaser.Math.Vector2(GameOptions.gameGravity, 0);
                this.jumpVector = new Phaser.Math.Vector2(-GameOptions.jumpForce, -walkingSpeed);
                break;
        }
    }

    private intersectOrthoSegment(move: Phaser.Geom.Line, seg: Phaser.Geom.Line): LineIntersection | null {
    
        if (seg.y1 === seg.y2) {
    
            const deltaY: number = move.y2 - move.y1;
            if (deltaY === 0) {
                return null;
            }
    
            const t1: number = (seg.y1 - move.y1) / deltaY;
            if (t1 < 0 || t1 > 1) {
                return null;
            }
    
            const hitX: number = move.x1 + (move.x2 - move.x1) * t1;
    
            const minX: number = Math.min(seg.x1, seg.x2);
            const maxX: number = Math.max(seg.x1, seg.x2);
    
            if (hitX < minX || hitX > maxX) {
                return null;
            }
    
            const t2: number = (seg.x2 !== seg.x1) ? (hitX - seg.x1) / (seg.x2 - seg.x1) : 0;
    
            return {
                point: new Phaser.Math.Vector2(hitX, seg.y1),
                t1: t1,
                t2: t2
            };
        }
    
        if (seg.x1 === seg.x2) {
    
            const deltaX: number = move.x2 - move.x1;
            if (deltaX === 0) {
                return null;
            }
    
            const t1: number = (seg.x1 - move.x1) / deltaX;
            if (t1 < 0 || t1 > 1) {
                return null;
            }
    
            const hitY: number = move.y1 + (move.y2 - move.y1) * t1;
    
            const minY: number = Math.min(seg.y1, seg.y2);
            const maxY: number = Math.max(seg.y1, seg.y2);
    
            if (hitY < minY || hitY > maxY) {
                return null;
            }
    
            const t2: number = (seg.y2 !== seg.y1) ? (hitY - seg.y1) / (seg.y2 - seg.y1) : 0;
    
            return {
                point: new Phaser.Math.Vector2(seg.x1, hitY),
                t1: t1,
                t2: t2
            };
        }
    
        return null;
    }
    
    private updateJump(dt: number): void {

        const prevX: number = this.hero.x;
        const prevY: number = this.hero.y;
    
        const nextX: number = prevX + this.jumpVector.x * dt;
        const nextY: number = prevY + this.jumpVector.y * dt;
    
        const movementLine: Phaser.Geom.Line = new Phaser.Geom.Line(prevX, prevY, nextX, nextY);
    
        let bestHit: LineIntersection | null = null;
        let bestPathIndex: number = -1;
        let bestSegmentIndex: number = -1;
        let bestT1: number = Number.POSITIVE_INFINITY;
    
        for (let i: number = 0; i < this.collisionPerimeterLines.length; i++) {
            for (let j: number = 0; j < this.collisionPerimeterLines[i].length; j++) {
    
                const realLine: Phaser.Geom.Line = this.collisionPerimeterLines[i][j];
                const hit: LineIntersection | null = this.intersectOrthoSegment(movementLine, realLine);
    
                if (hit !== null) {
                    if (this.skipCollision) {
                        this.skipCollision  =false;
                        continue;
                    }
    
                    if (hit.t1 < bestT1) {
                        bestT1 = hit.t1;
                        bestHit = hit;
                        bestPathIndex = i;
                        bestSegmentIndex = j;
                    }
                }
            }
        }
    
        if (bestHit !== null) {
            this.isJumping = false;
            const newSegment: MotionSegment = this.allPathSegments[bestPathIndex][bestSegmentIndex];
            const segDx: number = newSegment.end.x - newSegment.start.x;
            const segDy: number = newSegment.end.y - newSegment.start.y;
            const segLen: number = Math.hypot(segDx, segDy) || 1;
            const tx: number = segDx / segLen;
            const ty: number = segDy / segLen;
            const projectedSpeed: number = this.jumpVector.x * tx + this.jumpVector.y * ty;
            if (bestPathIndex !== this.pathIndex) {
                if (projectedSpeed > 0) {
                    this.direction = walkingDirection.CLOCKWISE;
	            }
	            else {
                    if (projectedSpeed < 0) {
		                this.direction = walkingDirection.COUNTERCLOCKWISE;
	                }
                }
            }

            this.pathIndex = bestPathIndex;
            this.segmentIndex = bestSegmentIndex;
            this.segmentT = bestHit.t2;
            this.hero.setPosition(bestHit.point.x, bestHit.point.y);
            return;
        }
    
        this.jumpVector.x += this.gravityVector.x * dt;
        this.jumpVector.y += this.gravityVector.y * dt;
    
        this.hero.x = nextX;
        this.hero.y = nextY;
    }
    
    update(_time: number, delta: number): void {

        if (this.isJumping) {
            this.updateJump(delta / 1000);
            return;
        }

	    const path: MotionSegment[] = this.allPathSegments[this.pathIndex];
        let pixelsToGo: number = GameOptions.heroSpeed * delta / 1000;
       
	    while (pixelsToGo > 0) {
            
            const segment: MotionSegment = path[this.segmentIndex];
            const segmentLength: number = segment.length * GameOptions.tileSize;
            const distanceLeft: number = this.direction === walkingDirection.CLOCKWISE ? segmentLength - this.segmentT * segmentLength : this.segmentT * segmentLength;
            
            if (pixelsToGo < distanceLeft) {
                this.segmentT += (this.direction === walkingDirection.CLOCKWISE ? 1 : -1) * pixelsToGo / segmentLength;
			    pixelsToGo = 0;
            }
		    else {
                pixelsToGo -= distanceLeft;
                if (this.direction === walkingDirection.CLOCKWISE) {
                    this.segmentIndex = (this.segmentIndex + 1) % path.length;
                    this.segmentT = 0;
                }
			    else {
                    this.segmentIndex = (this.segmentIndex - 1 + path.length) % path.length;
				    this.segmentT = 1;
			    }
		    }
	    }
	    this.placeOnSegment();
    }
}