/**
	=========================================================
	GREEDY RECTANGLE MERGING + PERIMETER EXTRACTION
	---------------------------------------------------------
	Pipeline:
	1) binary grid (0/1)
	2) greedy rectangle merging
	3) external perimeter extraction
	4) collinear segment merging
	5) closed loop construction

	Output:
	- rectangles
	- perimeter segments
	- ordered closed loops
	=========================================================
*/

/**
	Performs greedy rectangle merging on a binary grid.
*/
function greedyMerge(grid, mode) {

	const height = grid.length;
	const width = grid[0].length;

	const visited = new Array(height);
	for (let i = 0; i < height; i++) {
		visited[i] = new Array(width).fill(false);
	}

	const rectangles = [];

	for (let i = 0; i < height; i++) {
		for (let j = 0; j < width; j++) {

			if (visited[i][j] || grid[i][j] === 0) {
				visited[i][j] = true;
				continue;
			}

			let w = 1;	
            let h = 1;

			if (mode === 'horizontal-first') {

				while (j + w < width && !visited[i][j + w] && grid[i][j + w] === 1) {
					w++;
				}

				while (i + h < height) {
					let rowOk = true;

					for (let k = j; k < j + w; k++) {
						if (visited[i + h][k] || grid[i + h][k] === 0) {
							rowOk = false;
							break;
						}
					}

					if (!rowOk) {
                        break;
                    }
					h++;
				}

			} else {

				while (i + h < height && !visited[i + h][j] && grid[i + h][j] === 1) {
					h++;
				}

				while (j + w < width) {
					let colOk = true;

					for (let k = i; k < i + h; k++) {
						if (visited[k][j + w] || grid[k][j + w] === 0) {
							colOk = false;
							break;
						}
					}

					if (!colOk) {
                        break;
                    }
					w++;
				}
			}

			for (let k = i; k < i + h; k++) {
				for (let l = j; l < j + w; l++) {
					visited[k][l] = true;
				}
			}

			rectangles.push({ 
                x: j,
                y: i, 
                w, 
                h 
            });
		}
	}

	return rectangles;
}

/**
	Runs both greedy strategies and keeps the one
	that produces fewer rectangles.
*/
function greedyMergeBest(grid) {

	const horizontal = greedyMerge(grid, 'horizontal-first');
	const vertical = greedyMerge(grid, 'vertical-first');

	if (vertical.length < horizontal.length) {
		return { 
            rectangles: vertical, 
            mode: 'vertical-first'
        };
	}

	return {
        rectangles: horizontal, 
        mode: 'horizontal-first'
    }
}

/**
	Extracts the true external perimeter directly from a binary grid.

	Output:
	- array of segments { x1, y1, x2, y2 }
*/
function buildPerimeterFromGrid(grid) {

	const height = grid.length;
	const width = grid[0].length;
	const segments = [];

	function isFilled(i, j) {
		if (i < 0 || j < 0 || i >= height || j >= width) {
			return false;
		}
		return grid[i][j] === 1;
	}

	for (let i = 0; i < height; i++) {
		for (let j = 0; j < width; j++) {

			if (grid[i][j] === 0) {
                continue;
            }

			// top
			if (!isFilled(i - 1, j)) {
				segments.push({
                    x1: j,
                    y1: i,
                    x2: j + 1,
                    y2: i
                });
			}

			// bottom
			if (!isFilled(i + 1, j)) {
				segments.push({
                    x1: j,
                    y1: i + 1,
                    x2: j + 1,
                    y2: i + 1
                });
			}

			// left
			if (!isFilled(i, j - 1)) {
				segments.push({
                    x1: j,
                    y1: i,
                    x2: j,
                    y2: i + 1
                });
			}

			// right
			if (!isFilled(i, j + 1)) {
				segments.push({
                    x1: j + 1,
                    y1: i,
                    x2: j + 1,
                    y2: i + 1
                });
			}
		}
	}

	return segments;
}


/**
	Merges collinear and adjacent perimeter segments.
*/
function mergeCollinearSegments(segments) {

	const horizontal = new Map();
	const vertical = new Map();

	for (const s of segments) {
		if (s.y1 === s.y2) {
			if (!horizontal.has(s.y1)) {
                horizontal.set(s.y1, []);
            }
			horizontal.get(s.y1).push(s);
		} else {
			if (!vertical.has(s.x1)) {
                vertical.set(s.x1, []);
            }
			vertical.get(s.x1).push(s);
		}
	}

	const result = [];

	for (const list of horizontal.values()) {

		list.sort((a, b) => a.x1 - b.x1);
		let current = { ...list[0] };

		for (let i = 1; i < list.length; i++) {
			if (current.x2 === list[i].x1) {
				current.x2 = list[i].x2;
			} else {
				result.push(current);
				current = { ...list[i] };
			}
		}

		result.push(current);
	}

	for (const list of vertical.values()) {

		list.sort((a, b) => a.y1 - b.y1);
		let current = { ...list[0] };

		for (let i = 1; i < list.length; i++) {
			if (current.y2 === list[i].y1) {
				current.y2 = list[i].y2;
			} else {
				result.push(current);
				current = { ...list[i] };
			}
		}

		result.push(current);
	}

	return result;
}

/**
	Builds closed loops from perimeter segments.

	Output:
	- array of loops
	- each loop is an ordered array of points { x, y }
*/
function buildLoops(segments) {

	const adjacency = new Map();

	function key(p) {
		return `${p.x},${p.y}`;
	}

	function addEdge(a, b) {
		const ka = key(a);
		const kb = key(b);

		if (!adjacency.has(ka)) {
            adjacency.set(ka, []);
        }

		if (!adjacency.has(kb)) {
            adjacency.set(kb, []);
        }

		adjacency.get(ka).push(b);
		adjacency.get(kb).push(a);
	}

	for (const s of segments) {
		addEdge({ x: s.x1, y: s.y1 }, { x: s.x2, y: s.y2 });
	}

	const loops = [];
	const visitedEdges = new Set();

	function edgeKey(a, b) {
		return `${key(a)}-${key(b)}`;
	}

	for (const [startKey, neighbors] of adjacency.entries()) {

		for (const next of neighbors) {

			const start = parsePoint(startKey);
			const ek = edgeKey(start, next);
			if (visitedEdges.has(ek))  {
                continue;
            }

			const loop = [start];
			let current = start;
			let previous = null;

			while (true) {

				const candidates = adjacency.get(key(current));
				let chosen = null;

				for (const c of candidates) {
					if (previous && c.x === previous.x && c.y === previous.y) {
                        continue;
                    }
					chosen = c;
					break;
				}

				if (!chosen) {
                    break;
                }

				visitedEdges.add(edgeKey(current, chosen));
				visitedEdges.add(edgeKey(chosen, current));

				previous = current;
				current = chosen;

				if (current.x === start.x && current.y === start.y) {
					break;
				}

				loop.push(current);
			}

			if (loop.length > 2) {
				loops.push(loop);
			}
		}
	}

	return loops;
}

function parsePoint(key) {
	const [x, y] = key.split(',').map(Number);
	return { x, y };
}

/**
	High-level helper:
	grid → rectangles → loops
*/
function extractGreedyLoops(grid) {

	const result = greedyMergeBest(grid);

	const rawSegments = buildPerimeterFromGrid(grid);     

	const merged = mergeCollinearSegments(rawSegments);

	const loops = buildLoops(merged);

	return {
		mode: result.mode,
		rectangles: result.rectangles,
		segments: merged,
		loops
	};
}


export {
	greedyMerge,
	greedyMergeBest,
	buildPerimeterFromGrid,
	mergeCollinearSegments,
	buildLoops,
	extractGreedyLoops
};
