// THE GAME ITSELF
 
// modules to import
import { GameOptions } from '../gameOptions';
import { LevelThumbnail } from '../levelThumbnail';
import { PageSelector } from '../pageSelector';
 
// this class extends Scene class
export class PlayGame extends Phaser.Scene {
 
    // constructor
    constructor() {
        super({
            key: 'PlayGame'
        });
    }
 
    // method to be called once the class has been created
    create() : void {
 
        let isDragging: boolean = false;
        const pageSelectors: PageSelector[] = [];     
        let startX: number = 0;
        let startContainerX: number = 0;
        let currentPage: number = 0;

        const contentWidth: number = GameOptions.pages * this.scale.width;

        const container: Phaser.GameObjects.Container = this.add.container(0, 0);
        
        const scrollingMap: Phaser.GameObjects.TileSprite = this.add.tileSprite(0, 0, contentWidth, this.scale.height, 'transp');
        scrollingMap.setOrigin(0, 0);
        container.add(scrollingMap);

        const pageText: Phaser.GameObjects.Text = this.add.text(this.scale.width / 2, 16, 'Page 1 / ' + GameOptions.pages, {
            font : '18px Arial',
            color : '#ffffff',
            align : 'center'
        });
        pageText.setOrigin(0.5);

        const rowLength : number = GameOptions.thumbWidth * GameOptions.columns + GameOptions.spacing * (GameOptions.columns - 1);
        const columnHeight : number = GameOptions.thumbHeight * GameOptions.rows + GameOptions.spacing * (GameOptions.rows - 1);
        const leftMargin : number = (this.scale.width - rowLength) / 2 + GameOptions.thumbWidth / 2;  
        const topMargin : number = (this.scale.height - columnHeight) / 2 + GameOptions.thumbHeight / 2;
        let levelNumber: number = 0;
        for (let k : number = 0; k < GameOptions.pages; k ++) {
            for (let i : number = 0; i < GameOptions.rows; i ++) {
                for(let j : number = 0; j < GameOptions.columns; j ++) {
                    const posX : number = k * this.scale.width + leftMargin + j * (GameOptions.thumbWidth + GameOptions.spacing);
                    const posY : number = topMargin + i * (GameOptions.thumbHeight + GameOptions.spacing);
                    const thumb : LevelThumbnail = new LevelThumbnail(this, posX, posY, 'levelthumb', levelNumber + 1, k, levelNumber >= GameOptions.unlocked);
                    container.add(thumb)
                    thumb.on('levelSelected', (locked: boolean, level: number) => {
                        if (locked) {
                            alert('Level ' + level + ' is locked');
                        }
                        else {
                            alert('Play level ' + level);
                        }
                    });
                    levelNumber ++;
                }
            }
            pageSelectors[k] = new PageSelector(this, this.scale.width / 2 + (k - Math.floor(GameOptions.pages / 2) + 0.5 * (1 - GameOptions.pages % 2)) * 40, this.scale.height - 40, 'levelpages', k, 0);
            pageSelectors[k].on('pageSelected', (page: number) => {
                currentPage = page;
                const targetX: number = -currentPage * this.scale.width;
                this.tweens.add({
                    targets: container,
                    x: targetX,
                    duration: 250,
                    ease: 'Cubic.easeOut',
                    onComplete: () => {
                        snapToPage();    
                    }
                });
            });    
        }  

        this.input.on('pointerdown', (pointer: Phaser.Input.Pointer) => {
            isDragging = true;
            startX = pointer.x;
            startContainerX = container.x;
        });

        this.input.on('pointermove', (pointer: Phaser.Input.Pointer) => {
            if (isDragging) {
                const delta: number = pointer.x - startX;
                container.x = Phaser.Math.Clamp(startContainerX + delta, this.scale.width - contentWidth, 0);
            }
        });

        this.input.on('pointerup', () => {
            if (isDragging) {;
                isDragging = false;
                snapToPage();
            }
        });

        this.input.on('pointerupoutside', () => {
            if (isDragging) {
                isDragging = false;
                snapToPage();
            }
        });

        this.input.on('wheel', (_pointer: Phaser.Input.Pointer, _gameObjects: Phaser.GameObjects.GameObject[], deltaX: number, deltaY: number) => {
            const delta: number = (deltaX !== 0) ? deltaX : deltaY;
            if (delta != 0) {
                if (isDragging) {
                    isDragging = false;
                    startContainerX = container.x;
                }
                const direction: number = delta > 0 ? 1 : -1;
                const newPage: number = Phaser.Math.Clamp(currentPage + direction, 0, GameOptions.pages - 1);
                if (newPage != currentPage) {
                    currentPage = newPage;
                    moveToPage(currentPage);
                }
            }
        });

        const snapToPage = (): void => {
            const delta: number = startContainerX - container.x;
            currentPage = Math.round(-container.x / this.scale.width);
            if (Math.abs(delta) > this.scale.width * GameOptions.threshold) {
                if (delta > 0) {
                    currentPage = Math.ceil(-container.x / this.scale.width);
                }
                else {
                    currentPage = Math.floor(-container.x / this.scale.width);    
                }
            }
            currentPage = Phaser.Math.Clamp(currentPage, 0, GameOptions.pages - 1);
            moveToPage(currentPage);
        };

        const moveToPage = (page: number): void => {
            const targetX: number = -page * this.scale.width;
            this.tweens.add({
                targets: container,
                x: targetX,
                duration: 250,
                ease: 'Cubic.easeOut'
            });
            pageText.setText('Page ' + (currentPage + 1).toString() + ' / ' + GameOptions.pages);
            pageSelectors.forEach((selector) => {
                selector.updateThumb(currentPage)
            })
        }
    }  
}